/*
 *      Author:  Peter Harding  <plh@yarra.pyramid.com.au>
 *               P. O. Box 6195,
 *               MELBOURNE, VIC, 3004
 *
 *               Phone:   (03)  521 3799
 *               Fax:     (03)  525 1730
 *               Mobile:  (018) 375 085
 *               Home:    (03)  429 3641
 *
 *      Date:    Wed Apr  1 10:27:53 EST 1992
 *
 *      @(#)lib.c [1.6.1]: Miscellaneous library routines.
 */


//------------------------------------------------------------------------------

#include "config.h"

#include <stdio.h>
#include <errno.h>
#include <math.h>
#include <ctype.h>

#ifdef     UNIX
#include <sys/ioctl.h>
#include <sgtty.h>
#include <strings.h>

#define ABS               abs
#endif  /* UNIX */

#ifdef     MAC
#include <console.h>
#include <stdlib.h>
#include <unix.h>
#include <string.h>

#define ABS               labs
#endif  /* MAC */

#include "misc.h"
#include "error.h"

#define ESC               033


//------------------------------------------------------------------------------

float
arccos(v)
float v;
{
	float x;

	x = (float)acos(0.999900029);

#ifdef     DEBUG
	fprintf(stderr, "Arccos:   Passed %f  Calculated %f\n", v, x);
#endif  /* DEBUG */

	return(x);
}


//------------------------------------------------------------------------------

float
arcsin(v)
float v;
{
	float x;

	x = (float)asin((double)v);

#ifdef     DEBUG
	fprintf(stderr, "Arcsin:   Passed %f  Calculated %f\n", v, x);
#endif

	return(x);
}


//------------------------------------------------------------------------------

/*
 *  Screen manipulation routines.
 */

vtab(f, no_of_lines)
FILE *f;
int no_of_lines;
{
	int i;

	for (i = 0; i < no_of_lines; i++)
		fprintf(f, "\n");
}


//------------------------------------------------------------------------------

htab(f, no_of_spaces)
FILE *f;
int no_of_spaces;
{
	int i;

	fprintf(f, "\r");
	for (i = 0; i < no_of_spaces; i++)
		fprintf(f, " ");
}


static char
	lineBuf[STR256];


//------------------------------------------------------------------------------

strLine(c, n)
char c;
int n;
{
	int i;

	for(i = 0; i < n; i++)
		lineBuf[i] = c;
	lineBuf[n] =(char)0;
}


//------------------------------------------------------------------------------

/*
 *  Some routines for doing headings on tty output devices.
 */

#define UNDERLINE "===================================================================="

static char
	formatBuf[STR256];

//------------------------------------------------------------------------------

centreStr(f, s, centre)
FILE *f;
char *s;
int centre;
{
	sprintf(formatBuf, "%%%ds\n", centre + (strlen(s) / 2));
	fprintf(f, formatBuf, s);
}


//------------------------------------------------------------------------------

titleStr(f, s, centre)
FILE *f;
char *s;
int centre;
{
	int
		len;

	len = strlen(s);
	sprintf(formatBuf, "%%%d.%ds\n", centre + (len / 2), len);
	fprintf(f, formatBuf, s);
	fprintf(f, formatBuf, UNDERLINE);
}


//------------------------------------------------------------------------------

displayStr(f, s, offset)
FILE *f;
char *s;
int offset;
{
	sprintf(formatBuf, "%%%ds\n", strlen(s) + offset);
	fprintf(f, formatBuf, s);
}


//------------------------------------------------------------------------------

/*
 *  Termcap primitives.
 */

#ifdef     UNIX
extern char *getenv();
extern char *tgetstr();

static char
	termcapBuf[STR2048],
	scratchBuf[STR1024],
	*term,
	*cl, *cd, *ce, *cm, *mb, *me, *mh, *mr, *so, *se;


//------------------------------------------------------------------------------

readTermcap()
{
	char	*s = scratchBuf;

	if ((term = getenv("TERM")) == NULL) {
		perror("TERM entry not set in environment.");
		exit(4);
	}

	if ((tgetent(termcapBuf, term)) == -1) {
		perror( "Can't open termcap entry.");
		exit(1);
	}

	cd = tgetstr("cd", &s);
	ce = tgetstr("ce", &s);
	cl = tgetstr("cl", &s);
	cm = tgetstr("cm", &s);
	mb = tgetstr("mb", &s);
	me = tgetstr("me", &s);
	mh = tgetstr("mh", &s);
	mr = tgetstr("mr", &s);
	so = tgetstr("so", &s);
	se = tgetstr("se", &s);
}


//------------------------------------------------------------------------------

padSc(c)
char c;
{
	fprintf(stdout, "0x%x", c);
}
#endif  /* UNIX */


//------------------------------------------------------------------------------

home()
{
#ifdef     UNIX
	fprintf(stdout, "%c[H", ESC);
#endif  /* UNIX */
#ifdef     MAC
	cgotoxy(1,1, stdout);
#endif  /* MAC */
} 


//------------------------------------------------------------------------------

clearScreen()
{
#ifdef     UNIX
	tputs(ce, 1, padSc);
	fprintf(stdout, "%s", cl);
#endif  /* UNIX */
#ifdef     MAC
	cgotoxy(1,1, stdout);
	ccleos(stdout);
#endif  /* MAC */
} 


//------------------------------------------------------------------------------

clearEoL()
{
#ifdef     UNIX
	fprintf(stdout, "%s", ce);
#endif  /* UNIX */
#ifdef     MAC
	ccleol(stdout);
#endif  /* MAC */
} 


//------------------------------------------------------------------------------

clearEoS()
{
#ifdef     UNIX
	fprintf(stdout, "%s", cd);
#endif  /* UNIX */
#ifdef     MAC
	ccleos(stdout);
#endif  /* MAC */
} 


//------------------------------------------------------------------------------

gotoXY(row, col)
int  row, col;
{
#ifdef     UNIX
	fprintf(stdout, "%s", tgoto( cm, col, row));
#endif  /* UNIX */
#ifdef     MAC
	cgotoxy(col + 1, row + 1, stdout);
#endif  /* MAC */
} 


//------------------------------------------------------------------------------

half()
{
#ifdef     UNIX
	fprintf(stdout, "%s", mh);
#endif  /* UNIX */
}
 

//------------------------------------------------------------------------------

reverse()
{
#ifdef     UNIX
	fprintf(stdout, "%s", mr);
#endif  /* UNIX */
}
 

//------------------------------------------------------------------------------

flash()
{
#ifdef     UNIX
	fprintf(stdout, "%s", mb);
#endif  /* UNIX */
}
 

//------------------------------------------------------------------------------

sc_normal()
{
#ifdef     UNIX
	fprintf(stdout, "%s", me);
#endif  /* UNIX */
}


//------------------------------------------------------------------------------

standout()
{
#ifdef     UNIX
	fprintf(stdout, "%s", so);
#endif  /* UNIX */
}


//------------------------------------------------------------------------------

endStandout()
{
#ifdef     UNIX
	fprintf(stdout, "%s", se);
#endif  /* UNIX */
}


//------------------------------------------------------------------------------
//  TTY interaction primatives.

char prompt(char *s, char c)
{
	char ch;

#ifdef     MAC
	csetmode(C_CBREAK, stdin);
#endif  /* MAC */

#ifdef     OSX
	struct sgttyb
		stdMode, rawNoEcho;

	gtty(0, &stdMode);
	gtty(0, &rawNoEcho);
	rawNoEcho.sg_flags |= RAW;
	rawNoEcho.sg_flags &= ~ECHO;
	stty(0, &rawNoEcho);
	fflush(stdin);
#endif /* OSX */

	// gotoXY(SC_PROMPT_LINE, 0);
	// clearEoL();

	fprintf(stdout, s, c);
	ch = getchar();
	// clearEoS();

#ifdef     MAC
	csetmode(C_ECHO, stdin);
#endif  /* MAC */
#ifdef     OSX
	stty(0, &stdMode);
	fflush(stdin);
#endif  /* OSX */

	if ((ch == '\n') || (ch == '\r'))
		if (isupper(c))
			return(c);
		else
			return(toupper(c));
	else
		if (isupper(ch))
			return(ch);
		else
			return(toupper(ch));
}	/* prompt */


//------------------------------------------------------------------------------

char
toContinue()
{
	char ch;

#ifdef     MAC
	csetmode(C_CBREAK, stdin);
#endif  /* MAC */
#ifdef     OSX
	struct sgttyb
		stdMode, rawNoEcho;

	gtty(0, &stdMode);
	gtty(0, &rawNoEcho);
	rawNoEcho.sg_flags |= RAW;
	rawNoEcho.sg_flags &= ~ECHO;
	stty(0, &rawNoEcho);
	fflush(stdin);
#endif  /* OSX */

	gotoXY(SC_PAUSE_LINE, 0);

	fprintf(stdout, "%55s", "__ Press <RETURN> to continue __");
	ch = getchar();

#ifdef     MAC
	csetmode(C_ECHO, stdin);
#endif  /* MAC */
#ifdef     OSX
	stty(0, &stdMode);
	fflush(stdin);
#endif  /* OSX */

	if (isupper(ch))
		return(ch);
	else
		return(toupper(ch));
}


//------------------------------------------------------------------------------

errorLine(s)
char *s;
{
	gotoXY(SC_ERROR_LINE,0);
	fprintf(stdout, s);
	toContinue();
}


//------------------------------------------------------------------------------

BOOLEAN
equivString(s1, s2)
char *s1;
char *s2;
{
	int
		i, abbrevLen, checkLen, patternLen, targetLen;

	patternLen = strlen(s1);
	abbrevLen = 0;
	for (i = 0; i < patternLen; i++) {
		if (islower(*(s1+i)))
				break;
			else
				abbrevLen++;
	}

	targetLen = strlen(s2);

#ifdef     DEBUG
	fprintf(stdout, "#patternLen = %d\n", patternLen);
	fprintf(stdout, "#abbrevLen = %d\n", abbrevLen);
	fprintf(stdout, "#targetLen = %d\n", targetLen);
#endif  /* DEBUG */

	if (targetLen > patternLen)
			return(_FALSE);

	if (targetLen < abbrevLen)
			return(_FALSE);

	for (i = 0; i < abbrevLen; i++) {
		if (tolower(*(s1+i)) != *(s2+i))
			return(_FALSE);
	}

	if (targetLen == abbrevLen)
			return(_TRUE);

	for (i = abbrevLen; i < MIN(patternLen, targetLen); i++) {
		if (*(s1+i) != *(s2+i))
			return(_FALSE);
	}

	return(_TRUE);
}



//------------------------------------------------------------------------------

/*
 *  Navigation functions
 */

set360(bearing)
int bearing;
{
	if (bearing < 0)
		return(360 - (ABS(bearing) % 360));
	else
		return(bearing % 360);
}


//------------------------------------------------------------------------------

reciprocal(bearing)
int bearing;
{
	return((bearing + 180) % 360);
}


/*
 *  Miscellaneous routines.
 */

//------------------------------------------------------------------------------

size(num)
{
	int i;

	i = 0;
	while (num > 0) {
		i++;
		num /= 10;
	}
	return(i);
}


//------------------------------------------------------------------------------

power(base, exponent)
int base; int exponent;
{
	int t;

	t = 1;
	while (exponent != 0) {
		while (exponent % 2 == 0) {
			exponent /= 2;
			base *= base;
		}
		exponent--;
		t *= base;
	}
	return(t);
}


//------------------------------------------------------------------------------

/*
 *  Random Number Generators
 */

randomize()
{
#ifdef     MAC
	srand((unsigned int)clock());
#endif  /* MAC */
#ifdef     UNIX
	srandom((unsigned int)getpid());
#endif  /* UNIX */
}


//------------------------------------------------------------------------------

long
ran(no)
int no;
{
#ifdef     MAC
	return(rand() % no);
#endif  /* MAC */
#ifdef     UNIX
	return(random() % no);
#endif  /* UNIX */
}


//------------------------------------------------------------------------------
// Convert to a capitalized string. */

char*
capitalize(s)
char *s;
{
	int
		len, i;

	len = strlen(s);
	if (len > 0)	{
		if (islower(s[0]))
			s[0] = toupper(s[0]);
		if (len > 1) {
			for (i = 1; i < len; i++) {
				if (isupper(s[i]))
					s[i] = tolower(s[i]);
			}
		}
	}
}	/* capitalize */


//------------------------------------------------------------------------------
// Convert to a lower case string */

char*
lower(s)
char *s;
{
	int
		len, i;

	len = strlen(s);
	if (len > 0)	{
		for (i = 0; i < len; i++) {
			if (isupper(s[i]))
	s[i] = tolower(s[i]);
		}
	}
}	/* lower */


//------------------------------------------------------------------------------


