/*
 *     Author:    Peter Harding    <plh@pha.com.au>
 *                Peter Harding & Associates Pty. Ltd.
 *                Level 6,
 *                170 Queen Street,
 *                MELBOURNE, VIC, 3000
 *
 *                Phone:    03 9641 2222
 *                Fax:      03 9641 2200
 *                Mobile:   0418 375 085
 *
 *      Copyright (C) Peter Harding, 2002
 *
 *  $Id:$
 */

static char
    id[] = "@(#) [1.0.0] ntlm.c 23/05/2002";


//----------------------------------------------------------------------------

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <arpa/inet.h>

#include "misc.h"
#include "ntlm.h"
#include "md4.h"
#include "des.h"


//----------------------------------------------------------------------------

void dispno(char *filler, int no, uchar* u);


//----------------------------------------------------------------------------

void
dispno(
   char    *filler,
   int      no,
   uchar*   u
)
{
   int
      i;

   fprintf(stderr, "%s", filler);
   for (i = 0; i < no; i++)
      fprintf(stderr, "0x%02x ", u[ i ]);
   fprintf(stderr, " [Hex]\n");


   fprintf(stderr, "%s ", filler);
   for (i = 0; i < no; i++)
      fprintf(stderr, "%03o  ", u[ i ]);
   fprintf(stderr, "[Octal]\n");


   fprintf(stderr, "%s ", filler);
   for (i = 0; i < no; i++)
      fprintf(stderr, "%3d  ", u[ i ]);
   fprintf(stderr, "[Decimal]\n");
}  /* dispno */


//----------------------------------------------------------------------------

TYPE_1_MSG*
t1m_init(
   char *host,
   char *domain,
   int  *plength
)
{
   char
	   *sp;

   int
      ofd,
      dom_length,
      host_length,
      offset;

   TYPE_1_MSG_HDR
      hdr;

   static unsigned char
      msg[ 1024 ];

   host_length = strlen(host);
   dom_length  = strlen(domain);

   offset = sizeof(TYPE_1_MSG_HDR);

   fprintf(stderr, "Sizeof( TYPE_1_MSG_HDR) = %d\n", offset);

   memset((void *)&hdr, 0, offset);

   strncpy(hdr.protocol, "NTLMSSP", 7);

   hdr.type      = 0x01;
   hdr.flags     = LITTLE_ENDIAN(0xb203);

   hdr.host_length  = LITTLE_ENDIAN(host_length);
   hdr.host_length2 = LITTLE_ENDIAN(host_length);
   hdr.host_offset  = LITTLE_ENDIAN(offset);

   offset          += host_length;

   hdr.dom_length   = LITTLE_ENDIAN(dom_length);
   hdr.dom_length2  = LITTLE_ENDIAN(dom_length);
   hdr.dom_offset   = LITTLE_ENDIAN(offset);

   offset          += dom_length;

   memcpy((void*)&msg, (void*)&hdr, sizeof( hdr));

   sp  = msg + sizeof(hdr);

   strncpy(sp, (const char*)host, host_length);

   sp += host_length;

   strncpy(sp, (const char*)domain, dom_length);

   sp += dom_length;

   ofd = open("./t1m", O_RDWR | O_CREAT);
   write(ofd, &msg, offset);
   close(ofd);

   fprints(stderr, msg, offset, "<<<", 0);

   if (plength)
      *plength = offset;

   return (TYPE_1_MSG*)&msg;
}  /* t1m_init */


//----------------------------------------------------------------------------

void
disp_t1m(
   TYPE_1_MSG_HDR* phdr
)
{
   fprintf(stderr, "\n");
   fprintf(stderr, "Type 1 Message Header\n");
   fprintf(stderr, "=====================\n\n");

   fprintf(stderr, "       Protocol: ");
   fprints(stderr, phdr->protocol, 7, "<<<", 0);

   fprintf(stderr, "           Type: %d\n", phdr->type);
   fprintf(stderr, "          Flags: 0x%04x\n", phdr->flags);

   fprintf(stderr, " Domain Length1: 0x%04x\n", phdr->dom_length);
   fprintf(stderr, " Domain Length2: 0x%04x\n", phdr->dom_length2);
   fprintf(stderr, "  Domain Offset: 0x%04x\n", phdr->dom_offset);

   fprintf(stderr, "   Host Length1: 0x%04x\n", phdr->host_length);
   fprintf(stderr, "   Host Length2: 0x%04x\n", phdr->host_length2);
   fprintf(stderr, "    Host Offset: 0x%04x\n", phdr->host_offset);

   fprintf(stderr, "\n\n");
}  /* disp_t1m */


//----------------------------------------------------------------------------

TYPE_2_MSG*
t2m_init(
   char *nonce,
   int  *plength
)
{
   char
      *sp;

   int
      ofd,
      offset,
      nonce_length,
      msg_len       = 0x28;

   TYPE_2_MSG_HDR
      hdr;

   static unsigned char
      msg[ 1024 ];

   nonce_length  = strlen(nonce);

   if (nonce_length > 8)
      nonce[ 8 ] = '\0';

   offset = sizeof(TYPE_2_MSG_HDR);

   fprintf(stderr, "Sizeof( TYPE_2_MSG_HDR) = %d\n", offset);

   memset((void *)&hdr, 0, offset);

   strncpy(hdr.protocol, "NTLMSSP", 7);

   hdr.type      = 0x02;
   hdr.flags     = LITTLE_ENDIAN(0x8201);

   strncpy(hdr.nonce, nonce, 8);

   memcpy((void*)&msg, (void*)&hdr, sizeof( hdr));

   sp  = msg + sizeof(hdr);

   ofd = open("./t2m", O_RDWR | O_CREAT);
   write(ofd, &hdr, offset);
   close(ofd);

   fprints(stderr, msg, offset, "<<<", 0);

   if (plength)
      *plength = offset;

   return (TYPE_2_MSG*)&msg;
}  /* t2m_init */


//----------------------------------------------------------------------------

void
disp_t2m(
   TYPE_2_MSG_HDR* phdr
)
{
   char
      *sp;

   int
      i,
      length,
      offset;

   fprintf(stderr, "\n");
   fprintf(stderr, "Type 2 Message (%d)\n", sizeof(TYPE_2_MSG));
   fprintf(stderr, "=================\n\n");

   fprintf(stderr, "       Protocol: ");
   fprints(stderr, phdr->protocol, 7, "<<<", 0);

   fprintf(stderr, "           Type: %d\n", phdr->type);
   fprintf(stderr, "          Flags: 0x%04x\n", phdr->flags);
   fprintf(stderr, "     Msg Length: %d\n", LITTLE_ENDIAN( phdr->msg_len));
   fprintf(stderr, "       Length 1: %d\n", LITTLE_ENDIAN( phdr->length1));
   fprintf(stderr, "       Length 2: %d\n", LITTLE_ENDIAN( phdr->length2));
   fprintf(stderr, "      Unknown 1: %d\n", LITTLE_ENDIAN( phdr->unknown1));
   fprintf(stderr, "      Unknown 2: %d\n", LITTLE_ENDIAN( phdr->unknown2));
   fprintf(stderr, "      Unknown 3: %d\n", LITTLE_ENDIAN( phdr->unknown3));
   fprintf(stderr, "      Unknown 4: %d\n", LITTLE_ENDIAN( phdr->unknown4));
   fprintf(stderr, "      Unknown 5: %d\n", LITTLE_ENDIAN( phdr->unknown5));
   fprintf(stderr, "      Unknown 6: %d\n", LITTLE_ENDIAN( phdr->unknown6));

   fprintf(stderr, "          Nonce: ");

   dispno("                  ", 8, phdr->nonce);

   fprintf(stderr, "                  ");
   fprints(stderr, phdr->nonce, 8, "<<<", 0);

   fprintf(stderr, "\n");

   offset   = sizeof(TYPE_2_MSG_HDR);
   length   = LITTLE_ENDIAN(phdr->length2);
   length   = LITTLE_ENDIAN(phdr->unknown5);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), LITTLE_ENDIAN(phdr->unknown4));
   fprintf(stderr, "User:\n");
   fprints(stderr, sp, length, "<<<", 0);

   fprintf(stderr, "\n\n");
}  /* disp_t2m */



//----------------------------------------------------------------------------

char*
get_nonce(
   TYPE_2_MSG_HDR* pmsg
)
{
   static char
      nonce[ 9 ];

   strncpy(nonce, pmsg->nonce, 8);

   nonce[ 8 ] = '\0';

   return nonce;
}  /* get_nonce */


//----------------------------------------------------------------------------

TYPE_3_MSG*
t3m_init(
   char *user,
   char *host,
   char *domain,
   char *lm_resp,
   char *nt_resp,
   int  *plength
)
{
   char
      *sp;

   int
      ofd,
      user_length,
      host_length,
      dom_length,
      lm_resp_length,
      nt_resp_length,
      offset;

   TYPE_3_MSG_HDR
      hdr;

   static unsigned char
      msg[ 1024 ];

   user_length     = strlen(user);
   host_length     = strlen(host);
   dom_length      = strlen(domain);
   lm_resp_length  = strlen(lm_resp);
   nt_resp_length  = strlen(nt_resp);

   offset          = sizeof(TYPE_3_MSG_HDR);

   fprintf(stderr, "Sizeof( TYPE_3_MSG_HDR) = %d\n", offset);

   memset((void *)&hdr, 0, offset);

   strncpy(hdr.protocol, "NTLMSSP", 7);

   hdr.type            = 0x03;
   hdr.flags           = LITTLE_ENDIAN(0x8201);

   hdr.dom_length      = LITTLE_ENDIAN(dom_length);
   hdr.dom_length2     = LITTLE_ENDIAN(dom_length);
   hdr.dom_offset      = LITTLE_ENDIAN(offset);

   offset             += dom_length;

   hdr.user_length     = LITTLE_ENDIAN(user_length);
   hdr.user_length2    = LITTLE_ENDIAN(user_length);
   hdr.user_offset     = LITTLE_ENDIAN(offset);

   offset             += user_length;

   hdr.host_length     = LITTLE_ENDIAN(host_length);
   hdr.host_length2    = LITTLE_ENDIAN(host_length);
   hdr.host_offset     = LITTLE_ENDIAN(offset);

   offset             += host_length;

   hdr.lm_resp_length  = LITTLE_ENDIAN(lm_resp_length);
   hdr.lm_resp_length2 = LITTLE_ENDIAN(lm_resp_length);
   hdr.lm_resp_offset  = LITTLE_ENDIAN(offset);

   offset             += lm_resp_length;

   hdr.nt_resp_length  = LITTLE_ENDIAN(nt_resp_length);
   hdr.nt_resp_length2 = LITTLE_ENDIAN(nt_resp_length);
   hdr.nt_resp_offset  = LITTLE_ENDIAN(offset);

   offset             += nt_resp_length;

   memcpy((void*)&msg, (void*)&hdr, sizeof( hdr));

   sp  = msg + sizeof(hdr);

   strncpy(sp, (const char*)domain, dom_length);

   sp += dom_length;

   strncpy(sp, (const char*)user, user_length);

   sp += user_length;

   strncpy(sp, (const char*)host, host_length);

   sp += host_length;

   strncpy(sp, (const char*)lm_resp, lm_resp_length);

   sp += lm_resp_length;

   strncpy(sp, (const char*)nt_resp, nt_resp_length);

   sp += nt_resp_length;

   ofd = open("./t3m", O_RDWR | O_CREAT);
   write(ofd, &msg, offset);
   close(ofd);

   fprints(stderr, msg, ((int)sp - (int)&msg), "<<<", 0);

   if (plength)
      *plength = offset;

   return (TYPE_3_MSG*)&msg;
}  /* t3m_init */


//----------------------------------------------------------------------------

void
disp_t3m(
   TYPE_3_MSG_HDR* phdr
)
{
   char
      *sp;

   int
      length,
      offset;

   fprintf(stderr, "\n");
   fprintf(stderr, "Type 3 Message Header\n");
   fprintf(stderr, "=====================\n\n");

   fprintf(stderr, "       Protocol: ");
   fprints(stderr, phdr->protocol, 7, "<<<", 0);

   fprintf(stderr, "           Type: %d\n", phdr->type);
   fprintf(stderr, "          Flags: 0x%04x\n", LITTLE_ENDIAN( phdr->flags));

   fprintf(stderr, "   User Length1: 0x%04x\n", LITTLE_ENDIAN( phdr->user_length));
   fprintf(stderr, "   User Length2: 0x%04x\n", LITTLE_ENDIAN( phdr->user_length2));
   fprintf(stderr, "    User Offset: 0x%04x\n", LITTLE_ENDIAN( phdr->user_offset));

   fprintf(stderr, "   Host Length1: 0x%04x\n", LITTLE_ENDIAN( phdr->host_length));
   fprintf(stderr, "   Host Length2: 0x%04x\n", LITTLE_ENDIAN( phdr->host_length2));
   fprintf(stderr, "    Host Offset: 0x%04x\n", LITTLE_ENDIAN( phdr->host_offset));

   fprintf(stderr, " Domain Length1: 0x%04x\n", LITTLE_ENDIAN( phdr->dom_length));
   fprintf(stderr, " Domain Length2: 0x%04x\n", LITTLE_ENDIAN( phdr->dom_length2));
   fprintf(stderr, "  Domain Offset: 0x%04x\n", LITTLE_ENDIAN( phdr->dom_offset));

   fprintf(stderr, "LM Resp Length1: 0x%04x\n", LITTLE_ENDIAN( phdr->lm_resp_length));
   fprintf(stderr, "LM Resp Length2: 0x%04x\n", LITTLE_ENDIAN( phdr->lm_resp_length2));
   fprintf(stderr, " LM Resp Offset: 0x%04x\n", LITTLE_ENDIAN( phdr->lm_resp_offset));

   fprintf(stderr, "NT Resp Length1: 0x%04x\n", LITTLE_ENDIAN( phdr->nt_resp_length));
   fprintf(stderr, "NT Resp Length2: 0x%04x\n", LITTLE_ENDIAN( phdr->nt_resp_length2));
   fprintf(stderr, " NT Resp Offset: 0x%04x\n", LITTLE_ENDIAN( phdr->nt_resp_offset));

   fprintf(stderr, "\n");

   length   = LITTLE_ENDIAN(phdr->user_length);
   offset   = LITTLE_ENDIAN(phdr->user_offset);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), LITTLE_ENDIAN(phdr->user_offset));
   fprintf(stderr, "User:\n");
   fprints(stderr, sp, length, "<<<", 0);


   length   = LITTLE_ENDIAN(phdr->host_length);
   offset   = LITTLE_ENDIAN(phdr->host_offset);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), LITTLE_ENDIAN(phdr->host_offset));
   fprintf(stderr, "Host:\n");
   fprints(stderr, sp, length, "<<<", 0);


   length   = LITTLE_ENDIAN(phdr->dom_length);
   offset   = LITTLE_ENDIAN(phdr->dom_offset);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), LITTLE_ENDIAN(phdr->dom_offset));
   fprintf(stderr, "Domain:\n");
   fprints(stderr, sp, length, "<<<", 0);


   length   = LITTLE_ENDIAN(phdr->lm_resp_length);
   offset   = LITTLE_ENDIAN(phdr->lm_resp_offset);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), LITTLE_ENDIAN(phdr->lm_resp_offset));
   fprintf(stderr, "LM Response:\n");
   fprints(stderr, sp, length, "<<<", 0);


   length   = LITTLE_ENDIAN(phdr->nt_resp_length);
   offset   = LITTLE_ENDIAN(phdr->nt_resp_offset);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), LITTLE_ENDIAN(phdr->nt_resp_offset));
   fprintf(stderr, "NT Response:\n");
   fprints(stderr, sp, length, "<<<", 0);

   fprintf(stderr, "\n\n");
}  /* disp_t3m */


//----------------------------------------------------------------------------
// Create LanManager password

void
create_lm_passwd(
   const char  *passwd,
   uchar       *lm_passwd
)
{
   int
      i,
      len = strlen(passwd);

    if (len > 14)
      len = 14;

    for (i = 0; i < len; i++)
        lm_passwd[i] = toupper(passwd[i]);

    for (; i < 14; i++)
        lm_passwd[i] = 0;
}  /* create_lm_passwd */


//----------------------------------------------------------------------------
/* create LanManager hashed password */

void
create_lm_hashed_passwd(
   const char *lm_pw,
   uchar      *lm_hpw
)
{
   unsigned char
      magic[] = { 0x4B, 0x47, 0x53, 0x21, 0x40, 0x23, 0x24, 0x25 };
   
   des_key_schedule
      ks;

   des_setup(lm_pw,   ks);
   des_ecb_encrypt((des_cblock*)magic,  (des_cblock*)lm_hpw,   ks, DES_ENCRYPT);

   des_setup(lm_pw+7, ks);
   des_ecb_encrypt((des_cblock*)magic,  (des_cblock*)lm_hpw+8, ks, DES_ENCRYPT);

   memset(lm_hpw+16, 0, 5);
}  /* create_lm_hashed_passwd */


//----------------------------------------------------------------------------

void
create_nt_hashed_passwd(
   const char  *passwd,
   uchar       *nt_hpw
)
{
   int
      i,
      len = strlen(passwd);

   char
      nt_pw[2*len];

   MD4_CTX
      context;

   for (i = 0; i < len; i++)
   {
      nt_pw[ 2 * i ]     = passwd[ i ];
      nt_pw[ 2 * i + 1 ] = 0;
   }

   MD4Init(&context);

   MD4Update(&context, nt_pw, 2*len);

   MD4Final(nt_hpw, &context);

   memset(nt_hpw + 16, 0, 5);
}  /* create_nt_hashed_passwd */


//----------------------------------------------------------------------------

void
create_responses(
   char *nonce,
   char *lm_hpw,
   char *nt_hpw
)
{
   unsigned char
      lm_resp[24],
      nt_resp[24];

   des_calc(lm_hpw, nonce, lm_resp);
   des_calc(nt_hpw, nonce, nt_resp);
}  /* create_responses */


//----------------------------------------------------------------------------

void
build_t3m(
   char *user,
   char *passwd,
   char *host,
   char *domain,
   char *nonce
)
{
   char
      buf[ 1024 ],
      *sp;

   uchar
      lm_passwd[14],
      lm_hash[21],
      lm_resp[24],
      nt_hash[21],
      nt_resp[24];

   int
      mlen;

   create_lm_passwd(passwd, lm_passwd);

   fprintf(stderr, "LanManager Password:\n");
   fprints(stderr, lm_passwd, 14, 70, "<<<", 0);

   // lm_hash   = create_lm_hashed_passwd(lm_passwd);
   E_P16(lm_passwd, lm_hash);
   // memset(lm_hash + 16, 0, 5);

   fprintf(stderr, "LM Hashed Password:\n");
   fprints(stderr, lm_hash, 21, 70, "<<<", 0);

   create_nt_hashed_passwd(passwd, nt_hash);

   fprintf(stderr, "NT Hashed Password:\n");
   fprints(stderr, nt_hash, 21, 70, "<<<", 0);

   E_P24(lm_hash, nonce, lm_resp);
   E_P24(nt_hash, nonce, nt_resp);

   sp = (char*)base64encode(lm_resp, 0);

   fprintf(stderr, "Base64 Encoded LM Challenge Response:\n");
   fprintf(stderr, "%s\n", sp);

   sp = (char*)base64encode(nt_resp, 0);

   fprintf(stderr, "Base64 Encoded NT Challenge Response:\n");
   fprintf(stderr, "%s\n", sp);

   sp = (char*)t3m_init(user, host, domain, lm_resp, nt_resp, &mlen);

   sp = (char*)base64encode(sp, mlen);

   fprintf(stderr, "Base64 Encoded Type 3 Message: (length %d)\n", mlen);
   fprintf(stderr, "%s\n", sp);

   strcpy(buf, sp);

   fprintf(stderr, "Encoded length = %d\n", strlen( buf));

   sp = (char*)base64decode(buf, &mlen);

   fprintf(stderr, "Base64 Decoded Type 3 Message: (length %d)\n", mlen);
   fprints(stderr, sp, mlen, 70, "<<<", 0);

}  /* build_t3m */


//----------------------------------------------------------------------------

