/*
 *      Author:   Peter Harding  <plh@performiq.com.au>
 *                PerformIQ Pty. Ltd.
 *                Level 6,
 *                170 Queen Street
 *                MELBOURNE, VIC, 3000
 *
 *                Phone:   (03)  9641 2222
 *                Fax:     (03)  9641 2200
 *                Mobile:  0418 375 085
 *
 *      Copyright (C) Peter Harding, 2002
 *
 *  $Id:$
 */

static char
    id[] = "@(#) [1.0.0] ntlm.c 23/05/2002";


//----------------------------------------------------------------------------

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <arpa/inet.h>

#include "misc.h"
#include "ntlm.h"
#include "md4.h"
#include "base64.h"
#include "des.h"
//#include <openssl/des.h>


//----------------------------------------------------------------------------

int
   traceFlg    = 0;


//----------------------------------------------------------------------------

TYPE_1_MSG*
t1m_init(
   char *host,
   char *domain,
   int  *plength
)
{
   char
      uHost[128],
      uDomain[128],
      *sp;

   int
      ofd,
      dom_length,
      host_length,
      offset;

   TYPE_1_MSG_HDR
      hdr;

   static unsigned char
      msg[1024];

   if (host)
      host_length = unicode(uHost, host);
   else
      host_length = 0;

   if (domain)
      dom_length  = unicode(uDomain, domain);
   else
      dom_length  = 0;

   offset = sizeof(TYPE_1_MSG_HDR);

   // fprintf(stderr, "Sizeof(TYPE_1_MSG_HDR) = %d\n", offset);

   memset((void *)&hdr, 0, offset);

   strncpy(hdr.protocol, "NTLMSSP", 7);

   hdr.type      = 0x01;
   // hdr.flags     = TX_LITTLE_ENDIAN(0xb203);
   hdr.flags     = TX_LITTLE_ENDIAN(0x8207);
   hdr.unknown1  = TX_LITTLE_ENDIAN(0xa000);

   if (host_length) {
      hdr.host_length  = TX_LITTLE_ENDIAN(host_length);
      hdr.host_length2 = TX_LITTLE_ENDIAN(host_length);
      hdr.host_offset  = TX_LITTLE_ENDIAN(offset);

      offset          += host_length;
   }

   if (dom_length) {
      hdr.dom_length   = TX_LITTLE_ENDIAN(dom_length);
      hdr.dom_length2  = TX_LITTLE_ENDIAN(dom_length);
      hdr.dom_offset   = TX_LITTLE_ENDIAN(offset);

      offset          += dom_length;
   }

   memcpy((void*)&msg, (void*)&hdr, sizeof(hdr));

   sp  = msg + sizeof(hdr);

   if (host_length) {
      memcpy((void*)sp, (void*)uHost, host_length);

      sp += host_length;
   }

   if (dom_length) {
      memcpy((void*)sp, (void*)uDomain, dom_length);

      sp += dom_length;
   }

   if (traceFlg) {
      ofd = open("./t1m", O_RDWR | O_CREAT);
      write(ofd, &msg, offset);
      close(ofd);
   }

   // fprints(stderr, msg, offset, 70, "<<<", 1);

   if (plength)
      *plength = offset;

   return (TYPE_1_MSG*)&msg;
}	/* t1m_init */


//----------------------------------------------------------------------------

void
disp_t1m(
   TYPE_1_MSG* pmsg
)
{
   TYPE_1_MSG_HDR
      *phdr       = (TYPE_1_MSG_HDR*)pmsg;

   fprintf(stderr, "\n");
   fprintf(stderr, "Type 1 Message Header\n");
   fprintf(stderr, "=====================\n\n");

   fprintf(stderr, "       Protocol: ");
   fprints(stderr, phdr->protocol, 7, 70, "<<<", 1);

   fprintf(stderr, "           Type: %d\n", phdr->type);
   fprintf(stderr, "          Flags: 0x%04x\n", (unsigned short)phdr->flags);
   fprintf(stderr, "       Unknown1: 0x%04x\n", (unsigned short)phdr->unknown1);

   fprintf(stderr, " Domain Length1: 0x%04x\n", phdr->dom_length);
   fprintf(stderr, " Domain Length2: 0x%04x\n", phdr->dom_length2);
   fprintf(stderr, "  Domain Offset: 0x%04x\n", phdr->dom_offset);

   fprintf(stderr, "   Host Length1: 0x%04x\n", phdr->host_length);
   fprintf(stderr, "   Host Length2: 0x%04x\n", phdr->host_length2);
   fprintf(stderr, "    Host Offset: 0x%04x\n", phdr->host_offset);

   fprintf(stderr, "\n\n");
}	/* disp_t1m */


//----------------------------------------------------------------------------

TYPE_2_MSG*
t2m_init(
   char *nonce,
   int  *plength
)
{
   char
      payload[48],
      *sp;

   int
      ofd,
      offset,
      nonce_length,
      msg_len       = 0x28;

   TYPE_2_MSG_HDR
      hdr;

   static unsigned char
      msg[1024];

   nonce_length  = strlen(nonce);

   if (nonce_length > 8)
      nonce[8] = '\0';

   memset((void*)payload, 0, 48);

   // We will never use this in anger! - plh

   unicode(&payload[4], "EXCHANGE");
   unicode(&payload[24], "EDEN");
   unicode(&payload[36], "eden");

   payload[ 0] = 2;  // Type?
   payload[ 2] = 16; // Unicode length of 'EXCHANGE'
   payload[20] = 1;  // Type?
   payload[22] = 8;  // Unicode length of 'EDEN'
   payload[32] = 3;  // Type?
   payload[34] = 8;  // Unicode length of 'eden'

   fprintf(stderr, "Unicode payload is:\n");
   fprints(stderr, payload, 48, 70, "<<<", 1);
 
   offset = sizeof(TYPE_2_MSG_HDR);

   // fprintf(stderr, "Sizeof(TYPE_2_MSG_HDR) = %d\n", offset);

   memset((void *)&hdr, 0, offset);

   strncpy(hdr.protocol, "NTLMSSP", 7);

   hdr.type      = 0x02;
   // hdr.flags     = TX_LITTLE_ENDIAN(0x8201);

   strncpy(hdr.nonce, nonce, 8);

   memcpy((void*)&msg, (void*)&hdr, sizeof(hdr));

   sp  = msg + sizeof(hdr);

   if (traceFlg) {
      ofd = open("./t2m", O_RDWR | O_CREAT);
      write(ofd, &hdr, offset);
      close(ofd);
   }

   // fprints(stderr, msg, offset, 70, "<<<", 1);

   if (plength)
      *plength = offset;

   return (TYPE_2_MSG*)&msg;
}	/* t2m_init */


//----------------------------------------------------------------------------

void
disp_t2m(
   TYPE_2_MSG* pmsg
)
{
   char
      *sp;

   int
      i,
      length,
      offset;

   TYPE_2_MSG_HDR
      *phdr = (TYPE_2_MSG_HDR*)pmsg;

   fprintf(stderr, "\n");
   fprintf(stderr, "Type 2 Message (%d)\n", sizeof(TYPE_2_MSG));
   fprintf(stderr, "=================\n\n");

   fprintf(stderr, "       Protocol: ");
   fprints(stderr, phdr->protocol, 7, 70, "<<<", 1);

   fprintf(stderr, "           Type: %d\n", phdr->type);
   fprintf(stderr, "          Flags: 0x%04x\n", phdr->flags);
   fprintf(stderr, "   Dom Length 1: %d\n", TX_LITTLE_ENDIAN(phdr->dom_length));
   fprintf(stderr, "   Dom Length 2: %d\n", TX_LITTLE_ENDIAN(phdr->dom_length2));
   fprintf(stderr, "     Dom Offset: %d\n", TX_LITTLE_ENDIAN(phdr->dom_offset));
   fprintf(stderr, "      Unknown 1: %d\n", TX_LITTLE_ENDIAN(phdr->unknown1));
   fprintf(stderr, "      Unknown 2: %d\n", TX_LITTLE_ENDIAN(phdr->unknown2));
   fprintf(stderr, "   Pld Length 1: %d\n", TX_LITTLE_ENDIAN(phdr->pld_length));
   fprintf(stderr, "   Pld Length 1: %d\n", TX_LITTLE_ENDIAN(phdr->pld_length2));
   fprintf(stderr, "     Pld Offset: %d\n", TX_LITTLE_ENDIAN(phdr->pld_offset));
   fprintf(stderr, "      Unknown 3: %d\n", TX_LITTLE_ENDIAN(phdr->unknown3));

   fprintf(stderr, "          Nonce: ");

   dispno("                  ", 8, phdr->nonce);

   fprintf(stderr, "                  ");
   fprints(stderr, phdr->nonce, 8, 70, "<<<", 1);

   fprintf(stderr, "\n");

   offset   = sizeof(TYPE_2_MSG_HDR);
   sp       = (char*)((int)phdr + offset);
   length   = TX_LITTLE_ENDIAN(phdr->dom_length);

   fprintf(stderr, "Domain: (%d)\n", length);
   fprints(stderr, sp, length, 70, "<<<", 1);

   offset   = TX_LITTLE_ENDIAN(phdr->pld_offset);
   length   = TX_LITTLE_ENDIAN(phdr->pld_length);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), TX_LITTLE_ENDIAN(phdr->pld_length));
   fprintf(stderr, "Payload:\n");
   fprints(stderr, sp, length, 70, "<<<", 1);

   fprintf(stderr, "\n\n");
}	/* disp_t2m */



//----------------------------------------------------------------------------

char*
get_nonce(
   TYPE_2_MSG* pmsg
)
{
   static char
      nonce[9];

   TYPE_2_MSG_HDR
      *phdr        = (TYPE_2_MSG_HDR*)pmsg;

   memcpy((void*)nonce, (void*)phdr->nonce, 8);

   nonce[8] = '\0';

   return nonce;
}	/* get_nonce */


//----------------------------------------------------------------------------

TYPE_3_MSG*
t3m_init(
   char *user,
   char *host,
   char *domain,
   char *lm_resp,
   char *nt_resp,
   int  *plength
)
{
   char
      uUser[128],
      uHost[128],
      uDomain[128],
      *sp;

   int
      ofd,
      user_length,
      host_length,
      dom_length,
      lm_resp_length,
      nt_resp_length,
      offset;

   TYPE_3_MSG_HDR
      hdr;

   static unsigned char
      msg[1024];

   user_length     = unicode(uUser, user);
   host_length     = unicode(uHost, host);
   dom_length      = unicode(uDomain, domain);
   lm_resp_length  = 24;
   nt_resp_length  = 24;
   //lm_resp_length  = strlen(lm_resp);
   //nt_resp_length  = strlen(nt_resp);

   offset          = sizeof(TYPE_3_MSG_HDR);

   // fprintf(stderr, "Sizeof(TYPE_3_MSG_HDR) = %d\n", offset);

   memset((void *)&hdr, 0, offset);

   strncpy(hdr.protocol, "NTLMSSP", 7);

   hdr.type            = 0x03;
   hdr.flags           = TX_LITTLE_ENDIAN(0x8205);
   hdr.unknown3        = TX_LITTLE_ENDIAN(0x0000);
   hdr.unknown2        = TX_LITTLE_ENDIAN(0x0000);
   hdr.unknown1        = TX_LITTLE_ENDIAN(0xa080);

   hdr.dom_length      = TX_LITTLE_ENDIAN(dom_length);
   hdr.dom_length2     = TX_LITTLE_ENDIAN(dom_length);
   hdr.dom_offset      = TX_LITTLE_ENDIAN(offset);

   offset             += dom_length;

   hdr.user_length     = TX_LITTLE_ENDIAN(user_length);
   hdr.user_length2    = TX_LITTLE_ENDIAN(user_length);
   hdr.user_offset     = TX_LITTLE_ENDIAN(offset);

   offset             += user_length;

   hdr.host_length     = TX_LITTLE_ENDIAN(host_length);
   hdr.host_length2    = TX_LITTLE_ENDIAN(host_length);
   hdr.host_offset     = TX_LITTLE_ENDIAN(offset);

   offset             += host_length;

   hdr.lm_resp_length  = TX_LITTLE_ENDIAN(lm_resp_length);
   hdr.lm_resp_length2 = TX_LITTLE_ENDIAN(lm_resp_length);
   hdr.lm_resp_offset  = TX_LITTLE_ENDIAN(offset);

   offset             += lm_resp_length;

   hdr.nt_resp_length  = TX_LITTLE_ENDIAN(nt_resp_length);
   hdr.nt_resp_length2 = TX_LITTLE_ENDIAN(nt_resp_length);
   hdr.nt_resp_offset  = TX_LITTLE_ENDIAN(offset);

   offset             += nt_resp_length;

   hdr.msg_len         = TX_LITTLE_ENDIAN(offset);

   memcpy((void*)&msg, (void*)&hdr, sizeof(hdr));

   sp  = msg + sizeof(hdr);

   memcpy((void*)sp, (void*)uDomain, dom_length);

   sp += dom_length;

   memcpy((void*)sp, (void*)uUser, user_length);

   sp += user_length;

   memcpy((void*)sp, (void*)uHost, host_length);

   sp += host_length;

   memcpy((void*)sp, (void*)lm_resp, lm_resp_length);

   sp += lm_resp_length;

   memcpy((void*)sp, (void*)nt_resp, nt_resp_length);

   sp += nt_resp_length;

   if (traceFlg) {
      ofd = open("./t3m", O_RDWR | O_CREAT);
      write(ofd, &msg, offset);
      close(ofd);
   }

   // fprints(stderr, msg, ((int)sp - (int)&msg), 70, "<<<", 1);

   if (plength)
      *plength = offset;

   return (TYPE_3_MSG*)&msg;
}	/* t3m_init */


//----------------------------------------------------------------------------

void
disp_t3m(
   TYPE_3_MSG* pmsg
)
{
   char
      *sp;

   int
      length,
      offset;

   TYPE_3_MSG_HDR
      *phdr          = (TYPE_3_MSG_HDR*)pmsg;

   fprintf(stderr, "\n");
   fprintf(stderr, "Type 3 Message Header\n");
   fprintf(stderr, "=====================\n\n");

   fprintf(stderr, "       Protocol: ");
   fprints(stderr, phdr->protocol, 7, 70, "<<<", 1);

   fprintf(stderr, "           Type: %d\n", phdr->type);
   fprintf(stderr, "          Flags: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->flags));
   fprintf(stderr, "         MsgLen: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->msg_len));

   fprintf(stderr, "   User Length1: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->user_length));
   fprintf(stderr, "   User Length2: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->user_length2));
   fprintf(stderr, "    User Offset: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->user_offset));

   fprintf(stderr, "   Host Length1: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->host_length));
   fprintf(stderr, "   Host Length2: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->host_length2));
   fprintf(stderr, "    Host Offset: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->host_offset));

   fprintf(stderr, " Domain Length1: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->dom_length));
   fprintf(stderr, " Domain Length2: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->dom_length2));
   fprintf(stderr, "  Domain Offset: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->dom_offset));

   fprintf(stderr, "LM Resp Length1: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->lm_resp_length));
   fprintf(stderr, "LM Resp Length2: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->lm_resp_length2));
   fprintf(stderr, " LM Resp Offset: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->lm_resp_offset));

   fprintf(stderr, "NT Resp Length1: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->nt_resp_length));
   fprintf(stderr, "NT Resp Length2: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->nt_resp_length2));
   fprintf(stderr, " NT Resp Offset: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->nt_resp_offset));

   fprintf(stderr, "      Unknown 1: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->unknown1));
   fprintf(stderr, "      Unknown 2: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->unknown2));
   fprintf(stderr, "      Unknown 3: 0x%04x\n", TX_LITTLE_ENDIAN(phdr->unknown3));

   fprintf(stderr, "\n");

   length   = TX_LITTLE_ENDIAN(phdr->user_length);
   offset   = TX_LITTLE_ENDIAN(phdr->user_offset);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), TX_LITTLE_ENDIAN(phdr->user_offset));
   fprintf(stderr, "User:\n");
   fprints(stderr, sp, length, 70, "<<<", 1);


   length   = TX_LITTLE_ENDIAN(phdr->host_length);
   offset   = TX_LITTLE_ENDIAN(phdr->host_offset);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), TX_LITTLE_ENDIAN(phdr->host_offset));
   fprintf(stderr, "Host:\n");
   fprints(stderr, sp, length, 70, "<<<", 1);


   length   = TX_LITTLE_ENDIAN(phdr->dom_length);
   offset   = TX_LITTLE_ENDIAN(phdr->dom_offset);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), TX_LITTLE_ENDIAN(phdr->dom_offset));
   fprintf(stderr, "Domain:\n");
   fprints(stderr, sp, length, 70, "<<<", 1);


   length   = TX_LITTLE_ENDIAN(phdr->lm_resp_length);
   offset   = TX_LITTLE_ENDIAN(phdr->lm_resp_offset);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), TX_LITTLE_ENDIAN(phdr->lm_resp_offset));
   fprintf(stderr, "LM Response:\n");
   fprints(stderr, sp, length, 70, "<<<", 1);


   length   = TX_LITTLE_ENDIAN(phdr->nt_resp_length);
   offset   = TX_LITTLE_ENDIAN(phdr->nt_resp_offset);
   sp       = (char*)((int)phdr + offset);

   fprintf(stderr, "phdr  0x%08x  sp 0x%08x  delta %d (%d)\n",
      phdr, sp, ((int)sp - (int)phdr), TX_LITTLE_ENDIAN(phdr->nt_resp_offset));
   fprintf(stderr, "NT Response:\n");
   fprints(stderr, sp, length, 70, "<<<", 1);

   fprintf(stderr, "\n\n");
}	/* disp_t3m */


//----------------------------------------------------------------------------
// Create LanManager password

void
create_lm_passwd(
   const char  *passwd,
   uchar       *lm_passwd
)
{
   int
      i,
      len = strlen(passwd);

    if (len > 14)
      len = 14;

    for (i = 0; i < len; i++)
        lm_passwd[i] = toupper(passwd[i]);

    for (; i < 14; i++)
        lm_passwd[i] = 0;
}	/* create_lm_passwd */


//----------------------------------------------------------------------------
/* create LanManager hashed password */

void
create_lm_hashed_passwd(
   const char *lm_pw,
   uchar      *lm_hpw
)
{
   unsigned char
      magic[] = { 0x4B, 0x47, 0x53, 0x21, 0x40, 0x23, 0x24, 0x25 };
   
   des_key_schedule
      ks;

   des_setup(lm_pw,   ks);
   des_ecb_encrypt((des_cblock*)magic,  (des_cblock*)lm_hpw,   ks, DES_ENCRYPT);

   des_setup(lm_pw+7, ks);
   des_ecb_encrypt((des_cblock*)magic,  (des_cblock*)lm_hpw+8, ks, DES_ENCRYPT);

   memset(lm_hpw+16, 0, 5);
}	/* create_lm_hashed_passwd */


//----------------------------------------------------------------------------

void
create_nt_hashed_passwd(
   const char  *passwd,
   uchar       *nt_hpw
)
{
   int
      i,
      len = strlen(passwd);

   char
      nt_pw[len*2];

   MD4_CTX
      context;

   unicode(nt_pw, passwd);

   MD4Init(&context);

   MD4Update(&context, nt_pw, 2*len);

   MD4Final(nt_hpw, &context);

   memset(nt_hpw + 16, 0, 5);
}	/* create_nt_hashed_passwd */


//----------------------------------------------------------------------------

void
create_responses(
   char *nonce,
   char *lm_hpw,
   char *nt_hpw
)
{
   unsigned char
      lm_resp[24],
      nt_resp[24];

   des_calc(lm_hpw, nonce, lm_resp);
   des_calc(nt_hpw, nonce, nt_resp);
}	/* create_responses */


//----------------------------------------------------------------------------

TYPE_3_MSG*
build_t3m(
   char *user,
   char *passwd,
   char *host,
   char *domain,
   char *nonce,
   int  *plen,
   int   flg
)
{
   char
      buf[1024],
      *sp;

   uchar
      lm_passwd[14],
      lm_hash[21],
      lm_hash2[21],
      lm_resp[24],
      nt_hash[21],
      nt_hash2[21],
      nt_resp[24];

   int
      mlen,
      pwdlen;

   TYPE_3_MSG
      *pt3m;

   if (flg) {
      fprintf(stderr, "Password:\n");
      fprints(stderr, passwd, strlen(passwd), 70, "<<<", 1);
      fprintf(stderr, "Password Length: %d\n", strlen(passwd));
   }

   create_lm_passwd(passwd, lm_passwd);

   if (flg) {
      fprintf(stderr, "LanManager Password:\n");
      fprints(stderr, lm_passwd, sizeof(lm_passwd), 70, "<<<", 1);
      fprintf(stderr, "LM Password Length: %d\n", sizeof(lm_passwd));
   }


   // create_lm_hashed_passwd(lm_hash, lm_passwd);
   E_P16(lm_passwd, lm_hash);  // Use this instead

   memset(lm_hash + 16, 0, 5);

   if (flg) {
      fprintf(stderr, "LM Hashed Password:\n");
      fprints(stderr, lm_hash, 21, 70, "<<<", 1);
   }

   create_nt_hashed_passwd(passwd, nt_hash);

   if (flg) {
      fprintf(stderr, "NT Hashed Password:\n");
      fprints(stderr, nt_hash, 21, 70, "<<<", 1);
   }


/*
 * Added code to printout the actual responses, better comparing these
 * than attempting to compare the final base64 enc message or encoded
 * responses...
 */

   E_P24(lm_hash, nonce, lm_resp);

   if (flg) {
      fprintf(stderr, "LM Response:\n");
      fprints(stderr, lm_resp, 24, 70, "<<<", 1);

      sp = base64encode(lm_resp, 0);

      fprintf(stderr, "Base64 Encoded LM Challenge Response:\n");
      fprintf(stderr, "%s\n", sp);
   }

   E_P24(nt_hash, nonce, nt_resp);

   if (flg) {
      fprintf(stderr, "NT Respnse:\n");
      fprints(stderr, nt_resp, 24, 70, "<<<", 1);

      sp = base64encode(nt_resp, 0);

      fprintf(stderr, "Base64 Encoded NT Challenge Response:\n");
      fprintf(stderr, "%s\n", sp);
   }

   pt3m = t3m_init(user, host, domain, lm_resp, nt_resp, &mlen);

   if (flg) {
      sp = base64encode((char*)pt3m, mlen);

      fprintf(stderr, "Base64 Encoded Type 3 Message: (length %d)\n", mlen);
      fprintf(stderr, "Encoding: \"%s\"\n", sp);

      strcpy(buf, sp);

      fprintf(stderr, "Encoded length = %d\n", strlen(buf));

      sp = base64decode(buf, &mlen);

      fprintf(stderr, "Base64 Decoded Type 3 Message: (length %d)\n", mlen);
      fprints(stderr, sp, mlen, 70, "<<<", 1);
   }

   if (plen)
      *plen = mlen;

   return pt3m;
}	/* build_t3m */


//----------------------------------------------------------------------------

