

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>

#include "base64.h"


//----------------------------------------------------------------------------

#define BUF_LEN   1024

//----------------------------------------------------------------------------

static unsigned char
   dtable[256],                              // Encode/decode table
  *obuf              = (unsigned char*)0;    // Hook for outpur buffer

static int
	debugFlg          = 0,
	obuf_len          = 0;


/*****************************************************************************

  The procedure initialise_encoding_table()
  fills the binary encoding table with the characters
  the 6 bit values are mapped into.  The curious
  and disparate sequences used to fill this table
  permit this code to work both on ASCII and EBCDIC
  systems, the latter thanks to Ch.F.

  In EBCDIC systems character codes for letters are not
  consecutive; the initialisation must be split to accommodate
  the EBCDIC consecutive letters:

            A--I J--R S--Z a--i j--r s--z

  This code works on ASCII as well as EBCDIC systems.

*****************************************************************************/

static void
initialise_encoding_table(
   void
)
{
   int
      i;

   for (i = 0; i < 9; i++) {
      dtable[i]           = 'A' + i;
      dtable[i + 9]       = 'J' + i;
      dtable[26 + i]      = 'a' + i;
      dtable[26 + i + 9]  = 'j' + i;
   }

   for (i = 0; i < 8; i++) { 
      dtable[i + 18]      = 'S' + i;
      dtable[26 + i + 18] = 's' + i;
   }

   for (i = 0; i < 10; i++) {
      dtable[52 + i]      = '0' + i;
   }

   dtable[62]             = '+';
   dtable[63]             = '/';
}   /* initialise_encoding_table */


//----------------------------------------------------------------------------

char *
base64encode(
   unsigned char *ibuf,
	int            ilen
)
{
   int
      i,
		len,
		olen;

	unsigned char
		*ip,
		*op;

   initialise_encoding_table();

	if (!obuf) {
		if ((obuf = (unsigned char*)malloc(BUF_LEN)) == (unsigned char*)-1) {
			perror("malloc");
			exit(-1);
		}
		obuf_len = BUF_LEN;
   }

	*obuf = '\0';
	op    = obuf;

	if (!ibuf)
		return obuf;

	if ( ilen )
		len = ilen;
	else
		len = strlen( ibuf );

	olen = (strlen( ibuf ) / 3) * 4;

	if ( debugFlg ) {
  		fprintf( stderr, "obuf_len      -> %d\n", obuf_len );
  		fprintf( stderr, "strlen        -> %d\n", strlen( ibuf ));
  		fprintf( stderr, "len           -> %d\n", len);
  		fprintf( stderr, "olen          -> %d\n", olen);
	}

   if ( olen > obuf_len ) {  // realloc more space
   	olen = (((len / BUF_LEN) + 1) * BUF_LEN);
		if ((obuf = (unsigned char*)realloc(obuf, olen)) == (unsigned char*)-1) {
			perror("realloc");
			exit(-1);
		}
		obuf_len = olen;
	}

	if ( debugFlg ) {
  		fprintf( stderr, "obuf_len      -> %d\n", obuf_len );
	}

   ip = ibuf;

   while (ip < (ibuf + len)) {
      unsigned char
			igroup[3],
			ogroup[4];

      int
			c,
			n;

      igroup[0] = igroup[1] = igroup[2] = 0;

      for (n = 0; n < 3; n++) {
         c = *ip++;

			if ( ip <= (ibuf + len) )
         	igroup[n] = (unsigned char)c;
			else
				break;
      }

      if (n > 0) {
         ogroup[0] = dtable[igroup[0] >> 2];
         ogroup[1] = dtable[((igroup[0] & 3) << 4) | (igroup[1] >> 4)];
         ogroup[2] = dtable[((igroup[1] & 0xF) << 2) | (igroup[2] >> 6)];
         ogroup[3] = dtable[igroup[2] & 0x3F];

         /*
          * Replace characters in output stream with "=" pad
          * characters if fewer than three characters were
          * read from the end of the input stream.
          */

         if (n < 3) {
            ogroup[3] = '=';

            if (n < 2) {
               ogroup[2] = '=';
            }
         }

         for (i = 0; i < 4; i++) {
            *op++ = ogroup[i];
         }
      }
   }

   *op = '\0';

	return obuf;
}   /* base64encode */


/*****************************************************************************

  Procedure initialise_decode_table() creates the lookup table
  used to map base64 characters into their binary values from
  0 to 63.  The table is built in this rather curious way in
  order to be properly initialised for both ASCII-based
  systems and those using EBCDIC, where the letters are not
  contiguous.  (EBCDIC fixes courtesy of Ch.F.)

  In EBCDIC systems character codes for letters are not
  consecutive; the initialisation must be split to accommodate
  the EBCDIC consecutive letters:

   A--I J--R S--Z a--i j--r s--z

  This code works on ASCII as well as EBCDIC systems.

*****************************************************************************/

static void
initialise_decode_table(
   void
)
{
	int
		i;

   for (i = 0; i < 255; i++) {
      dtable[i]    = 0x80;
   }

   for (i = 'A'; i <= 'I'; i++) {
      dtable[i]    = 0 + (i - 'A');
   }

   for (i = 'J'; i <= 'R'; i++) {
      dtable[i]    = 9 + (i - 'J');
   }                      

   for (i = 'S'; i <= 'Z'; i++) {
      dtable[i]    = 18 + (i - 'S');
   }                      

   for (i = 'a'; i <= 'i'; i++) {
      dtable[i]    = 26 + (i - 'a');
   }

   for (i = 'j'; i <= 'r'; i++) {
      dtable[i]    = 35 + (i - 'j');
   }

   for (i = 's'; i <= 'z'; i++) {
      dtable[i]    = 44 + (i - 's');
   }

   for (i = '0'; i <= '9'; i++) {
      dtable[i]    = 52 + (i - '0');
   }

   dtable['+']     = 62;
   dtable['/']     = 63;
   dtable['=']     = 0;
}   /* initialise_decode_table */


//----------------------------------------------------------------------------

char *
base64decode(
   unsigned char *ibuf,
	int           *plen
)
{
   int
		decodedLength   = 0,
		errcheck        = 0,
      i,
		len,
		no;

	unsigned char
		*ip,
		*op;

   initialise_decode_table();

	if ( !obuf ) {
		if ((obuf = (unsigned char*)malloc(BUF_LEN)) == (unsigned char*)-1) {
			perror("malloc");
			exit(-1);
		}

		obuf_len = BUF_LEN;
   }

	*obuf = '\0';
   op    = obuf;

	if (!ibuf)
		return obuf;

	len = (strlen( ibuf ) / 4) * 3;

	if ( debugFlg ) {
  		fprintf( stderr, "obuf_len      -> %d\n", obuf_len );
  		fprintf( stderr, "strlen        -> %d\n", strlen( ibuf ));
  		fprintf( stderr, "len           -> %d\n", len);
	}

   if ( len > obuf_len ) {  // realloc more space
   	len = (((len / BUF_LEN) + 1) * BUF_LEN);
		if ((obuf = (unsigned char*)realloc(obuf, len)) == (unsigned char*)-1) {
			perror("realloc");
			exit(-1);
		}
		obuf_len = len;
	}

	if ( debugFlg ) {
  		fprintf( stderr, "obuf_len      -> %d\n", obuf_len );
	}

   ip = ibuf;

   while (*ip) {
      unsigned char
          a[4], b[4], o[3];

      for (i = 0; i < 4; i++) {
         int
            c = *ip++;

         if (!c) {
            if (errcheck && (i > 0)) {
               fprintf(stderr, "[base64decode]  Input string incomplete.\n");
					*op = '\0';
            	return obuf;
            }
         }

         if (dtable[c] & 0x80) {
            if (errcheck) {
               fprintf(stderr, "[base64decode]  Illegal character '%c' in input file.\n", c);
					*op = '\0';
            	return obuf;
            }
            /* Ignoring errors: discard invalid character. */
            i--;
            continue;
         }

         a[i] = (unsigned char) c;
         b[i] = (unsigned char) dtable[c];
      }

      o[0] = (b[0] << 2) | (b[1] >> 4);
      o[1] = (b[1] << 4) | (b[2] >> 2);
      o[2] = (b[2] << 6) | b[3];

      no = a[2] == '=' ? 1 : (a[3] == '=' ? 2 : 3);

		decodedLength += no;

		for ( i = 0; i < no; i++ )
			*op++ = o[i];

      if (i < 3) {
         break;
      }
   }

	*op = '\0';

	if ( plen )
		*plen = decodedLength;

   return obuf;
}   /* base64decode */


//----------------------------------------------------------------------------

